<?php

namespace App\Rules;

use Illuminate\Contracts\Validation\Rule;

class CheckValidFile implements Rule
{
    /**
     * Set error message.
     *
     * @param string $errorMessage
     * @var string
     */
    protected $errorMessage;

    /**
     * Allowed file extention.
     *
     * @var array
     */
    protected $allowFile;

    /**
     * Check maximum file size
     *
     * @var bool
     */
    protected $checkMaxFileSize;

    /**
     * Set $allowFile and $checkMaxFileSize.
     *
     * @param  array  $allowFile
     * @param  bool  $checkMaxFileSize
     * @return void
     */
    public function __construct(array $allowFile = [], bool $checkMaxFileSize = true)
    {
        $this->allowFile = $allowFile;
        $this->checkMaxFileSize = $checkMaxFileSize;
    }

    /**
     * Determine if the validation rule passes.
     *
     * @param string $attribute
     * @param  mixed  $value
     * @return bool
     */
    public function passes($attribute, $value)
    {
        if ($value === null || !method_exists($value, 'getClientOriginalExtension')) {
            $this->errorMessage = __('The file is required.');
            return false;
        }
        
        $isValid = in_array($value->getClientOriginalExtension(), $this->allowFile);
        if (!$isValid) {
            $this->errorMessage = __('Allowed File Extensions: ') . implode(", ", $this->allowFile);
            return false;
        }

        if ($this->checkMaxFileSize) {
           
            $maxFileSize = maxFileSize($_FILES[$attribute]["size"] ?? $value->getSize());
            if ($maxFileSize['status'] == 0) {
                $this->errorMessage = $maxFileSize['message'];
                return false;
            }
        }

        return true;
    }

    /**
     * Get the validation error message.
     *
     * @return string
     */
    public function message()
    {
        return $this->errorMessage;
    }
}
